<?php
session_start();
header('Content-Type: application/json');

function jsonResponse($success, $message = '', $data = []) {
    echo json_encode(array_merge(['success' => $success, 'message' => $message], $data));
    exit;
}

try {
    $db = new PDO('sqlite:../ibo_panel.db');
    $db->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

    $input = json_decode(file_get_contents('php://input'), true);
    $action = $input['action'] ?? 'view';

    switch ($action) {
        case 'view':
            $stmt = $db->query("SELECT * FROM playlist");
            $playlistRecords = $stmt->fetchAll(PDO::FETCH_ASSOC);
            jsonResponse(true, '', ['data' => $playlistRecords]);
            break;

        case 'get_dns_options':
            $stmt = $db->query("SELECT id, title FROM dns");
            $dnsOptions = $stmt->fetchAll(PDO::FETCH_ASSOC);
            jsonResponse(true, '', ['data' => $dnsOptions]);
            break;

        case 'add':
            $dns_id = filter_var($input['dns_id'] ?? '', FILTER_SANITIZE_NUMBER_INT);
            $mac_address = trim($input['mac_address'] ?? '');
            $username = trim($input['username'] ?? '');
            $password = trim($input['password'] ?? '');
            $pin = trim($input['pin'] ?? '');

            if (!$dns_id || !$mac_address || !$username || !$password || !$pin) {
                jsonResponse(false, 'All fields are required and must be valid.');
            }

            // Opcional: validar formato MAC (simples)
            if (!preg_match('/^([0-9A-Fa-f]{2}[:-]){5}([0-9A-Fa-f]{2})$/', $mac_address)) {
                jsonResponse(false, 'Invalid MAC address format.');
            }

            $stmt = $db->prepare("INSERT INTO playlist (dns_id, mac_address, username, password, pin) VALUES (:dns_id, :mac_address, :username, :password, :pin)");
            $stmt->execute([
                ':dns_id' => $dns_id,
                ':mac_address' => $mac_address,
                ':username' => $username,
                ':password' => $password,
                ':pin' => $pin,
            ]);

            $newId = $db->lastInsertId();
            jsonResponse(true, 'Playlist entry added successfully.', ['new_id' => $newId]);
            break;

        case 'edit':
            $id = filter_var($input['id'] ?? 0, FILTER_VALIDATE_INT);
            $dns_id = filter_var($input['dns_id'] ?? '', FILTER_SANITIZE_NUMBER_INT);
            $mac_address = trim($input['mac_address'] ?? '');
            $username = trim($input['username'] ?? '');
            $password = trim($input['password'] ?? '');
            $pin = trim($input['pin'] ?? '');

            if (!$id || !$dns_id || !$mac_address || !$username || !$password || !$pin) {
                jsonResponse(false, 'All fields are required and must be valid.');
            }

            if (!preg_match('/^([0-9A-Fa-f]{2}[:-]){5}([0-9A-Fa-f]{2})$/', $mac_address)) {
                jsonResponse(false, 'Invalid MAC address format.');
            }

            $stmt = $db->prepare("UPDATE playlist SET dns_id = :dns_id, mac_address = :mac_address, username = :username, password = :password, pin = :pin WHERE id = :id");
            $stmt->execute([
                ':dns_id' => $dns_id,
                ':mac_address' => $mac_address,
                ':username' => $username,
                ':password' => $password,
                ':pin' => $pin,
                ':id' => $id,
            ]);

            if ($stmt->rowCount() === 0) {
                jsonResponse(false, 'No playlist entry found with the given ID.');
            }

            jsonResponse(true, 'Playlist entry updated successfully.');
            break;

        case 'delete':
            $id = filter_var($input['id'] ?? 0, FILTER_VALIDATE_INT);
            if (!$id) {
                jsonResponse(false, 'Invalid ID for deletion.');
            }

            $stmt = $db->prepare("DELETE FROM playlist WHERE id = :id");
            $stmt->execute([':id' => $id]);

            if ($stmt->rowCount() === 0) {
                jsonResponse(false, 'No playlist entry found with the given ID.');
            }

            jsonResponse(true, 'Playlist entry deleted successfully.');
            break;

        default:
            jsonResponse(false, 'Invalid action.');
    }
} catch (PDOException $e) {
    jsonResponse(false, 'Database error: ' . $e->getMessage());
}
